<?php

namespace App\Services;

use App\DTOs\Guest\UpdateEventDTO;
use App\Models\Event;
use Exception;
use Illuminate\Support\Facades\Log;

class EventService
{
    public function get(): ?Event
    {
        return Event::first();
    }

    public function find(int $id): ?Event
    {
        return Event::find($id);
    }

    public function create(array $data): ?Event
    {
        try {
            return Event::create($data);
        } catch (Exception $e) {
            Log::error('Error al crear evento: ' . $e->getMessage(), [
                'data' => $data,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function update(int $id, array $data): ?Event
    {
        try {
            $event = Event::find($id);
            if ($event) {
                $event->update($data);
            }
            return $event;
        } catch (Exception $e) {
            Log::error('Error al actualizar evento: ' . $e->getMessage(), [
                'id' => $id,
                'data' => $data,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function upsert(UpdateEventDTO $dto): ?Event
    {
        $data = $dto->toArray();
        try {
            $event = Event::firstOrNew(); // si no existe, crea una nueva instancia (no guardada)
            $event->fill($data);         // llena con los datos
            $event->save();              // guarda (create o update según corresponda)
            return $event;
        } catch (Exception $e) {
            Log::error('Error al guardar evento: ' . $e->getMessage(), [
                'data' => $data,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function delete(int $id): bool
    {
        $event = Event::find($id);
        return $event ? $event->delete() : false;
    }
}